<?php

namespace App\Http\Controllers\Gateway;

use App\Constants\Status;
use App\Http\Controllers\Controller;
use App\Lib\FormProcessor;
use App\Models\AdminNotification;
use App\Models\Advertisement;
use App\Models\Deposit;
use App\Models\GatewayCurrency;
use App\Models\PurchasedVideo;
use App\Models\Transaction;
use App\Models\User;
use App\Models\UserNotification;
use App\Models\Video;
use Illuminate\Http\Request;

class PaymentController extends Controller {
    public function deposit($id = null, $monetization = null) {

        if ($monetization && !gs('monetization_status')) {
            abort(404);
        }

        $gatewayCurrency = GatewayCurrency::whereHas('method', function ($gate) {
            $gate->where('status', Status::ENABLE);
        })->with('method')->orderby('name')->get();
        $advertisement = '';
        if ($id) {
            $advertisement = Advertisement::where('user_id', auth()->id())->pending()->findOrFail($id);
        }

        $pageTitle = 'Payment Methods';

        return view('Template::user.payment.deposit', compact('gatewayCurrency', 'pageTitle', 'advertisement', 'monetization'));
    }

    public function depositInsert(Request $request) {

        $request->validate([
            'amount'       => 'required|numeric|gt:0',
            'gateway'      => 'required',
            'currency'     => 'required',
            'video_id'     => 'sometimes|integer',
            'monetization' => 'nullable|in:1',
        ]);

        if ($request->monetization && !gs('monetization_status')) {
            abort(404);
        }

        $video         = null;
        $advertisement = null;

        $user = auth()->user();

        if ($request->video_id) {
            $video  = Video::published()->stock()->findOrFail($request->video_id);
            $amount = $video->price;
        } else if ($request->advertisement_id) {
            $advertisement = Advertisement::where('user_id', $user->id)->pending()->findOrFail($request->advertisement_id);
            $amount        = $advertisement->total_amount;
        } else if ($request->monetization) {
            $amount = gs('monetization_amount');
        } else {
            $amount = $request->amount;
        }

        $gate = GatewayCurrency::whereHas('method', function ($gate) {
            $gate->where('status', Status::ENABLE);
        })->where('method_code', $request->gateway)->where('currency', $request->currency)->first();
        if (!$gate) {
            $notify[] = ['error', 'Invalid gateway'];
            return back()->withNotify($notify);
        }

        if ($gate->min_amount > $amount || $gate->max_amount < $amount) {
            $notify[] = ['error', 'Please follow payment limit'];
            return back()->withNotify($notify);
        }

        $charge      = $gate->fixed_charge + ($amount * $gate->percent_charge / 100);
        $payable     = $amount + $charge;
        $finalAmount = $payable * $gate->rate;

        $data                  = new Deposit();
        $data->user_id         = $user->id;
        $data->method_code     = $gate->method_code;
        $data->method_currency = strtoupper($gate->currency);
        $data->amount          = $amount;
        $data->charge          = $charge;
        $data->rate            = $gate->rate;
        $data->final_amount    = $finalAmount;
        $data->btc_amount      = 0;
        $data->btc_wallet      = "";
        $data->trx             = getTrx();

        if ($video) {
            $data->video_id    = $video->id;
            $data->success_url = route('video.play', [$video->id, $video->slug]);
            $data->failed_url  = route('video.play', [$video->id, $video->slug]);
        } else if ($advertisement) {
            $data->advertisement_id = $advertisement->id;
            $data->success_url      = route('user.advertiser.ad.list');
            $data->failed_url       = route('user.advertiser.ad.list');
        } else if ($request->monetization) {
            $data->is_monetization = Status::YES;
            $data->success_url     = route('user.monetization');
            $data->failed_url      = route('user.monetization');
        }

        $data->save();
        session()->put('Track', $data->trx);
        return to_route('user.deposit.confirm');
    }

    public function depositConfirm() {
        $track   = session()->get('Track');
        $deposit = Deposit::where('trx', $track)->where('status', Status::PAYMENT_INITIATE)->orderBy('id', 'DESC')->with('gateway')->firstOrFail();

        if ($deposit->method_code >= 1000) {
            return to_route('user.deposit.manual.confirm');
        }

        $dirName = $deposit->gateway->alias;
        $new     = __NAMESPACE__ . '\\' . $dirName . '\\ProcessController';

        $data = $new::process($deposit);
        $data = json_decode($data);

        if (isset($data->error)) {
            $notify[] = ['error', $data->message];
            return back()->withNotify($notify);
        }
        if (isset($data->redirect)) {
            return redirect($data->redirect_url);
        }

        // for Stripe V3
        if (@$data->session) {
            $deposit->btc_wallet = $data->session->id;
            $deposit->save();
        }
        $advertisement = false;
        if ($deposit->advertisement_id) {
            $advertisement = true;
        }

        $pageTitle = 'Payment Confirm';
        return view("Template::$data->view", compact('data', 'pageTitle', 'deposit', 'advertisement'));
    }

    public static function userDataUpdate($deposit, $isManual = null) {
        if ($deposit->status == Status::PAYMENT_INITIATE || $deposit->status == Status::PAYMENT_PENDING) {
            $deposit->status = Status::PAYMENT_SUCCESS;
            $deposit->save();

            $user = User::find($deposit->user_id);
            if (!$deposit->video_id) {
                $user->balance += $deposit->amount;
                $user->save();

            } else if ($deposit->video_id) {
                $videoOwner = User::find($deposit->video->user_id);
                $videoOwner->balance += $deposit->amount;
                $videoOwner->save();
            }

            $methodName = $deposit->methodName();

            $transaction               = new Transaction();
            $transaction->user_id      = $deposit->user_id;
            $transaction->amount       = $deposit->amount;
            $transaction->post_balance = $user->balance;
            $transaction->charge       = $deposit->charge;
            $transaction->trx_type     = '+';
            $transaction->details      = 'Payment Via ' . $methodName;
            $transaction->trx          = $deposit->trx;
            $transaction->remark       = 'deposit';
            $transaction->save();

            if (!$isManual) {
                $adminNotification            = new AdminNotification();
                $adminNotification->user_id   = $user->id;
                $adminNotification->title     = 'Payment successful via ' . $methodName;
                $adminNotification->click_url = urlPath('admin.deposit.successful');
                $adminNotification->save();
            }

            if ($deposit->video_id) {
                $transaction               = new Transaction();
                $transaction->user_id      = $deposit->user_id;
                $transaction->video_id     = $deposit->video_id;
                $transaction->amount       = $deposit->amount;
                $transaction->post_balance = $user->balance;
                $transaction->charge       = $deposit->charge;
                $transaction->trx_type     = '-';
                $transaction->details      = 'Payment Via ' . $methodName;
                $transaction->trx          = $deposit->trx;
                $transaction->remark       = 'purchased_video';
                $transaction->save();

                $transaction               = new Transaction();
                $transaction->user_id      = $videoOwner->id;
                $transaction->video_id     = $deposit->video_id;
                $transaction->amount       = $deposit->amount;
                $transaction->post_balance = $videoOwner->balance;
                $transaction->charge       = 0;
                $transaction->trx_type     = '+';
                $transaction->details      = $deposit->user->fullname . " purchased your video.";
                $transaction->trx          = $deposit->trx;
                $transaction->remark       = 'earn_from_video';
                $transaction->save();

                $purchased           = new PurchasedVideo();
                $purchased->user_id  = $deposit->user_id;
                $purchased->video_id = $deposit->video_id;
                $purchased->owner_id = $deposit->video->user_id;
                $purchased->trx      = $deposit->trx;
                $purchased->amount   = $deposit->amount;
                $purchased->save();

                $adminNotification            = new AdminNotification();
                $adminNotification->user_id   = $user->id;
                $adminNotification->title     = 'Payment successful via ' . $methodName;
                $adminNotification->click_url = urlPath('admin.deposit.successful');
                $adminNotification->save();

                $userNotification            = new UserNotification();
                $userNotification->user_id   = $deposit->video->user_id;
                $userNotification->title     = 'New Purchase: ' . $deposit->video->title;
                $userNotification->click_url = urlPath('video.play', [$deposit->video->id, $deposit->video->slug]);
                $userNotification->save();

            } else if ($deposit->advertisement_id) {
                $advertisement                 = $deposit->advertisement;
                $advertisement->status         = Status::RUNNING;
                $advertisement->payment_status = Status::PAYMENT_SUCCESS;
                $advertisement->save();

                $transaction               = new Transaction();
                $transaction->user_id      = $deposit->user_id;
                $transaction->amount       = $deposit->amount;
                $transaction->post_balance = $user->balance;
                $transaction->charge       = $deposit->charge;
                $transaction->trx_type     = '-';
                $transaction->details      = 'Payment Via ' . $methodName;
                $transaction->trx          = $deposit->trx;
                $transaction->remark       = 'advertisements_published';
                $transaction->save();

                $adminNotification            = new AdminNotification();
                $adminNotification->user_id   = $user->id;
                $adminNotification->title     = 'Payment successful via ' . $methodName;
                $adminNotification->click_url = urlPath('admin.deposit.successful');
                $adminNotification->save();

                $userNotification            = new UserNotification();
                $userNotification->user_id   = $deposit->user_id;
                $userNotification->title     = $deposit->advertisement->title . " has been published.";
                $userNotification->click_url = urlPath('user.advertiser.ad.list');
                $userNotification->save();

            } else if ($deposit->is_monetization) {
                $user                      = $deposit->user;
                $user->monetization_status = Status::MONETIZATION_APPROVED;
                $user->save();

                $transaction               = new Transaction();
                $transaction->user_id      = $deposit->user_id;
                $transaction->amount       = $deposit->amount;
                $transaction->post_balance = $user->balance;
                $transaction->charge       = $deposit->charge;
                $transaction->trx_type     = '-';
                $transaction->details      = 'Payment Via ' . $methodName;
                $transaction->trx          = $deposit->trx;
                $transaction->remark       = 'purchased_monetization';
                $transaction->save();

                $adminNotification            = new AdminNotification();
                $adminNotification->user_id   = $user->id;
                $adminNotification->title     = 'Payment successful via ' . $methodName;
                $adminNotification->click_url = urlPath('admin.deposit.successful');
                $adminNotification->save();

                $userNotification            = new UserNotification();
                $userNotification->user_id   = $deposit->user_id;
                $userNotification->title     = "Payment successfully captured.";
                $userNotification->click_url = urlPath('user.monetization');
                $userNotification->save();
            }

            notify($user, $isManual ? 'DEPOSIT_APPROVE' : 'DEPOSIT_COMPLETE', [
                'method_name'     => $methodName,
                'method_currency' => $deposit->method_currency,
                'method_amount'   => showAmount($deposit->final_amount, currencyFormat: false),
                'amount'          => showAmount($deposit->amount, currencyFormat: false),
                'charge'          => showAmount($deposit->charge, currencyFormat: false),
                'rate'            => showAmount($deposit->rate, currencyFormat: false),
                'trx'             => $deposit->trx,
                'post_balance'    => showAmount($user->balance),
            ]);

            if ($deposit->video_id) {
                notify($user, 'PURCHASED_VIDEO', [
                    'title'           => $deposit->video->title,
                    'method_name'     => $methodName,
                    'method_currency' => $deposit->method_currency,
                    'method_amount'   => showAmount($deposit->final_amount, currencyFormat: false),
                    'amount'          => showAmount($deposit->amount, currencyFormat: false),
                    'charge'          => showAmount($deposit->charge, currencyFormat: false),
                    'rate'            => showAmount($deposit->rate, currencyFormat: false),
                    'trx'             => $deposit->trx,
                    'post_balance'    => showAmount($user->balance),
                ]);

            }

            if ($deposit->advertisement_id) {
                notify($user, 'PUBLISHED_ADVERTISEMENT', [
                    'title'           => $deposit->advertisement->title,
                    'click'           => $deposit->advertisement->click,
                    'impression'      => $deposit->advertisement->impression,
                    'method_name'     => $methodName,
                    'method_currency' => $deposit->method_currency,
                    'method_amount'   => showAmount($deposit->final_amount, currencyFormat: false),
                    'amount'          => showAmount($deposit->amount, currencyFormat: false),
                    'charge'          => showAmount($deposit->charge, currencyFormat: false),
                    'rate'            => showAmount($deposit->rate, currencyFormat: false),
                    'trx'             => $deposit->trx,
                    'post_balance'    => showAmount($user->balance),
                ]);

            }

            if ($deposit->is_monetization) {
                notify($user, 'PURCHASED_MONETIZATION', [
                    'method_name'     => $methodName,
                    'method_currency' => $deposit->method_currency,
                    'method_amount'   => showAmount($deposit->final_amount, currencyFormat: false),
                    'amount'          => showAmount($deposit->amount, currencyFormat: false),
                    'charge'          => showAmount($deposit->charge, currencyFormat: false),
                    'rate'            => showAmount($deposit->rate, currencyFormat: false),
                    'trx'             => $deposit->trx,
                ]);

            }

        }
    }

    public function manualDepositConfirm() {
        $track = session()->get('Track');
        $data  = Deposit::with('gateway')->where('status', Status::PAYMENT_INITIATE)->where('trx', $track)->first();
        abort_if(!$data, 404);
        if ($data->method_code > 999) {
            $pageTitle = 'Confirm Payment';
            $method    = $data->gatewayCurrency();
            $gateway   = $method->method;
            return view('Template::user.payment.manual', compact('data', 'pageTitle', 'method', 'gateway'));
        }
        abort(404);
    }

    public function manualDepositUpdate(Request $request) {
        $track = session()->get('Track');
        $data  = Deposit::with('gateway')->where('status', Status::PAYMENT_INITIATE)->where('trx', $track)->first();
        abort_if(!$data, 404);
        $gatewayCurrency = $data->gatewayCurrency();
        $gateway         = $gatewayCurrency->method;
        $formData        = $gateway->form->form_data;

        $formProcessor  = new FormProcessor();
        $validationRule = $formProcessor->valueValidation($formData);
        $request->validate($validationRule);
        $userData = $formProcessor->processFormData($request, $formData);

        $data->detail = $userData;
        $data->status = Status::PAYMENT_PENDING;
        $data->save();

        $advertisement = $data->advertisement;

        if ($advertisement) {
            $advertisement->payment_status = Status::PAYMENT_PENDING;
            $advertisement->status         = Status::PAUSE;
            $advertisement->save();
        }

        if ($data->is_monetization) {
            $user                      = $data->user;
            $user->monetization_status = Status::MONETIZATION_APPLYING;
            $user->save();

        }

        $adminNotification            = new AdminNotification();
        $adminNotification->user_id   = $data->user->id;
        $adminNotification->title     = 'Payment request from ' . $data->user->username;
        $adminNotification->click_url = urlPath('admin.deposit.details', $data->id);
        $adminNotification->save();

        notify($data->user, 'DEPOSIT_REQUEST', [
            'method_name'     => $data->gatewayCurrency()->name,
            'method_currency' => $data->method_currency,
            'method_amount'   => showAmount($data->final_amount, currencyFormat: false),
            'amount'          => showAmount($data->amount, currencyFormat: false),
            'charge'          => showAmount($data->charge, currencyFormat: false),
            'rate'            => showAmount($data->rate, currencyFormat: false),
            'trx'             => $data->trx,
        ]);

        $notify[] = ['success', 'You have payment request has been taken'];
        return redirect($data->success_url)->withNotify($notify);

    }

}
